#if NUNIT
using System;
using System.Data;
using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;

using gov.va.med.vbecs.UnitTests;

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	[TestFixture]
	public class Parsers_CprsCancelVbecsOrder
	{
		private const string SELECT_PENDING_ORDER_TEST_STATEMENT = "SELECT TOP 1 OrderedTest.CprsOrderNumber, OrderableTest.CprsOrderableItemId FROM OrderedTest INNER JOIN OrderableTest ON OrderedTest.OrderableTestId = OrderableTest.OrderableTestId WHERE OrderStatusCode = 'P'";
		private const string SELECT_PENDING_ORDER_COMPONENT_STATEMENT = "SELECT TOP 1 OrderedComponent.CprsOrderNumber, ComponentClass.CprsOrderableItemId FROM OrderedComponent INNER JOIN ComponentClass ON OrderedComponent.ComponentClassId = ComponentClass.ComponentClassId WHERE OrderStatusCode = 'P'";
		private const string TAS_TEST_ORDER = "1^TAS^99VBC^^";
		private const string RBC_COMPONENT_ORDER = "2^RED BLOOD CELLS^99VBC^^";
		private const string KC_DIVISION = "589^VA HEARTLAND - WEST, VISN 15";
		private const string CARRIAGE_RETURN = "\x0D";

		private string _componentMessage;
		private string _testMessage;

		[SetUp]
		public void SetUp()
		{
			Common.LogonUser.LogonUserDUZ = "53311";
			Common.LogonUser.LogonUserDivisionCode = UnitTestConfig.TestDivisionCode;
			//
			Guid [] guids = DataCreator.CreateNewOrder();
			Assert.IsTrue( guids != null && guids.Length == 2 );
			//
			string componentQuery = 
				"SELECT TOP 1 OC.CprsOrderNumber, OC.VbecsOrderNumber, PO.OrderGroupNumber, P.VistaPatientId, P.PatientSsn, P.PatientLastName, P.PatientFirstName " +
				"FROM OrderedComponent OC " +
				"JOIN PatientOrder PO ON OC.PatientOrderGuid = OC.PatientOrderGuid " +
				"JOIN PatientTreatment PT ON PO.PatientTreatmentGuid = PT.PatientTreatmentGuid " +
				"JOIN Patient P ON PT.PatientGuid = P.PatientGuid " +
				"WHERE OrderedComponentGuid = '" + guids[0] + "'";
			//
			DataRow drComponent = UnitTestUtil.RunSQLGetDataRow( componentQuery );
			Assert.IsNotNull( drComponent );
			//
			_componentMessage = 	
				"MSH|^~\\&|OERR|589|VBECS|589|20110412135954-0500||OMG^O19|5894298274|T|2.4|||AL|AL|USA" + CARRIAGE_RETURN +
				"PID|||^^^^NI~" + drComponent["PatientSsn"].ToString() + "^^^^SS~" + drComponent["VistaPatientId"].ToString() + "^^^^PI||" + drComponent["PatientLastName"].ToString() + "^" + drComponent["PatientFirstName"].ToString() + "||19240608|F" + CARRIAGE_RETURN +
				"PV1||O|KC-1010 NEW PT CLINIC 2|||||||" + CARRIAGE_RETURN +
				"ORC|CA|" + drComponent["CprsOrderNumber"].ToString() + "^OR|1116^VBEC|" + drComponent["OrderGroupNumber"].ToString() + "^OR|||||201104121359-0500|53355^ONE^USER||53355^ONE^USER||||^Entered in error|589^VA HEARTLAND - WEST, VISN 15|||" + CARRIAGE_RETURN +
				"OBR|1|" + drComponent["CprsOrderNumber"].ToString() + "^OR|1116^VBEC|2^RED BLOOD CELLS^99VBC^^|||||||L||1||^^||||||||||||1^^^^201104121359-0500^R^^" + CARRIAGE_RETURN;
			//
			//
			string testQuery = 
				"SELECT TOP 1 OT.CprsOrderNumber, OT.VbecsOrderNumber, PO.OrderGroupNumber, P.VistaPatientId, P.PatientSsn, P.PatientLastName, P.PatientFirstName " +
				"FROM OrderedTest OT " +
				"JOIN PatientOrder PO ON OT.PatientOrderGuid = OT.PatientOrderGuid " +
				"JOIN PatientTreatment PT ON PO.PatientTreatmentGuid = PT.PatientTreatmentGuid " +
				"JOIN Patient P ON PT.PatientGuid = P.PatientGuid " +
				"WHERE OrderedTestGuid = '" + guids[1] + "'";
			//
			DataRow drTest = UnitTestUtil.RunSQLGetDataRow( testQuery );
			Assert.IsNotNull( drTest );
			//
			_testMessage = 	
				"MSH|^~\\&|OERR|589|VBECS|589|20110412135955-0500||OMG^O19|5894298276|T|2.4|||AL|AL|USA" + CARRIAGE_RETURN +
				"PID|||^^^^NI~" + drTest["PatientSsn"].ToString() + "^^^^SS~" + drTest["VistaPatientId"].ToString() + "^^^^PI||" + drTest["PatientLastName"].ToString() + "^" + drTest["PatientFirstName"].ToString() + "||19240608|F" + CARRIAGE_RETURN +
				"PV1||O|KC-1010 NEW PT CLINIC 2|||||||" + CARRIAGE_RETURN +
				"ORC|CA|" + drTest["CprsOrderNumber"].ToString() + "^OR|1117^VBEC|" + drTest["OrderGroupNumber"].ToString() + "^OR|||||201104121359-0500|53355^ONE^USER||53355^ONE^USER||||^Entered in error|589^VA HEARTLAND - WEST, VISN 15|||" + CARRIAGE_RETURN +
				"OBR|1|" + drTest["CprsOrderNumber"].ToString() + "^OR|1117^VBEC|1^TYPE \\T\\ SCREEN^99VBC^^|||||||O||1||^^SEND PATIENT||||||||||||^^^20110412^201104121359-0500^R^^" + CARRIAGE_RETURN;
		}

		/// <summary>
		/// Ctor - PASS (Ordered Component)
		/// </summary>
		[Test]
		public void Ctor_Pass_Ordered_Component()
		{
			Assert.IsNotNull( new CprsCancelVbecsOrder(new HL7OmgMessage(_componentMessage)) );
		}

		/// <summary>
		/// Ctor - PASS (Ordered Test)
		/// </summary>
		[Test]
		public void Ctor_Pass_Ordered_Test()
		{
			Assert.IsNotNull( new CprsCancelVbecsOrder(new HL7OmgMessage(_testMessage)) );
		}

		/// <summary>
		/// Ctor - FAIL (null message)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void Ctor_Fail_Null_Message()
		{
			CprsCancelVbecsOrder nulException = new CprsCancelVbecsOrder( null );
		}

		/// <summary>
		/// ProcessCancelOrder - PASS (Component Order)
		/// </summary>
		[Test]
		public void ProcessCancelOrder_Pass_Ordered_Component()
		{
			HL7OrgMessage ackMessage = new CprsCancelVbecsOrder( new HL7OmgMessage(_componentMessage) ).ProcessCancelOrder();
			if( ackMessage.AcknowledgementCode != AckCodes.AA.ToString() )
				Assert.Fail( "Acknowledgement Code is " + ackMessage.AcknowledgementCode + " and should be AA.");
		}

		/// <summary>
		/// ProcessCancelOrder - PASS (Test Order)
		/// </summary>
		[Test]
		public void ProcessCancelOrder_Pass_Ordered_Test()
		{
			HL7OrgMessage ackMessage = new CprsCancelVbecsOrder( new HL7OmgMessage(_testMessage) ).ProcessCancelOrder();
			if( ackMessage.AcknowledgementCode != AckCodes.AA.ToString() )
				Assert.Fail( "Acknowledgement Code is " + ackMessage.AcknowledgementCode + " and should be AA.");
		}
	}
}
#endif